module.exports = function (grunt) {
  'use strict';

  var requirejs = require('requirejs');
  var path = require('path');

  var rDefineStart = /define\([^{]*?{/;
  var rDefineEndWithReturn = /\s*return\s+[^\}]+(\}\);[^\w\}]*)$/;
  var rDefineEnd = /\}\);[^}\w]*$/;

  var optimize = function(data, callback) {
    requirejs.optimize({
      name: 'summernote/summernote',
      baseUrl: data.baseUrl,
      /**
       * Handle final output from the optimizer
       */
      out: function (compiled) {
        // 01. Embed version
        var version = grunt.config('pkg.version');
        compiled = compiled.replace(/@VERSION/g, version);

        // 02. Output file
        compiled = compiled.replace(/@FILENAME/g, path.basename(data.outFile));

        // 02.  Embed Date
        var date = (new Date()).toISOString().replace(/:\d+\.\d+Z$/, 'Z');
        compiled = compiled.replace(/@DATE/g, date);

        grunt.file.write(data.outFile, compiled);
      },
      optimize: 'none',
      wrap: {
        startFile: path.join(data.baseUrl, data.startFile),
        endFile: path.join(data.baseUrl, data.endFile)
      },
      findNestedDependencies: true,
      skipSemiColonInsertion: true,
      /**
       * Strip all definitions generated by requirejs
       *
       * @param {String} name
       * @param {String} path
       * @param {String} contents The contents to be written (including their AMD wrappers)
       */
      onBuildWrite: function (name, path, contents) {
        contents = contents.replace(rDefineStart, '');

        if (rDefineEndWithReturn.test(contents)) {
          contents = contents.replace(rDefineEndWithReturn, '');
        } else {
          contents = contents.replace(rDefineEnd, '');
        }
        return contents;
      },
      excludeShallow: ['jquery', 'codemirror', 'app'],
      include: [data.entryFile],
      paths: {
        jquery: 'empty:',
        codemirror: 'empty:'
      },
      packages: [{
        name: 'summernote',
        location: './'
      }]
    }, function () {
      callback();
    }, function (err) {
      callback(err);
    });
  };

  grunt.registerMultiTask('build', 'concatenate source: summernote.js', function () {
    var done = this.async();
    var counter = this.data.length;

    this.data.forEach(function(datum) {
      optimize(datum, function (err) {
        counter -= 1;

        if (err) {
          grunt.log.error(err);
        }

        if (counter == 0) {
          done();
        }
      });
    });
  });
};
